<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Session;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Inertia\Response;

class ProfessionalController extends Controller
{
    /**
     * Display the professionals list.
     */
    public function index(Request $request): Response
    {
        $professionals = User::where('user_type_id', 2)->paginate(config('app.pagination'));

        return Inertia::render('Professional/Index', [
            'professionals' => $professionals,
            'success_msg' => Session::has('success_msg') ? Session::get('success_msg') : null,
            'error_msg' => Session::has('error_msg') ? Session::get('error_msg') : null
        ]);
    }

    /**
     * Display the professional create form.
     */
    public function create(Request $request): Response
    {
        return Inertia::render('Professional/Create');
    }

    /**
     * Display the professional edit form.
     */
    public function edit(Request $request, $id): Response
    {
        $professional = User::find($id);

        return Inertia::render('Professional/Update', [
            'professional' => $professional
        ]);
    }

    /**
     * Delete the professional .
     */
    public function delete(Request $request, $id)
    {
        $professional = User::find($id);

        if ($professional->appointments->count() > 0) {
            return to_route('professional.index')->with('error_msg', 'Professional cannot be deleted, because this user has appointments');
        }

        return DB::transaction(function () use ($professional) {

            $professional->delete();
    
            return to_route('professional.index')->with('success_msg', 'Professional has been deleted successfully');
        });

    }

    /**
     * Save new professional.
     */
    public function save(Request $request)
    {
        $data = $request->validate([
            'name' => ['required', 'string', 'max:100'],
            'contact' => ['required', 'numeric', 'unique:users,contact'],
            'email' => ['required', 'email', 'unique:users,email'],
            'password' => ['required', 'string', 'min:8', 'confirmed']
        ]);

        $user_params = Arr::except( $request->all(), ['password_confirmation']);
        $user_params['password'] = Hash::make($user_params['password']);
        $user_params['user_type_id'] = 2;

        return DB::transaction(function () use ($user_params) {
            $user = User::create($user_params);

            return to_route('professional.index')->with('success_msg', 'Professional has been created successfully');
        });
    }

    /**
     * Update new user.
     */
    public function update(Request $request)
    {
        $data = $request->validate([
            'id' => ['required', 'numeric'],
            'name' => ['required', 'string', 'max:100'],
            'contact' => ['required', 'numeric', 'unique:users,contact,'.$request->id],
            'email' => ['required', 'email', 'unique:users,email,'.$request->id],
        ]);

        $professional = User::find($request->id);
        $professional_params = $request->all();


        return DB::transaction(function () use ($professional_params, $professional) {
            $professional->update($professional_params);
            return to_route('professional.index')->with('success_msg', 'Professional has been updated successfully');
        });
    }
}
